# analysis/rg_flow_analysis.py

import numpy as np
import matplotlib.pyplot as plt


def load_rg_flow(path: str = "results/rg_flow.npy") -> np.ndarray:
    """
    Load the RG‑flow array from disk.

    Args:
      path: Path to the NumPy .npy file produced by run_rg_flow.py

    Returns:
      A 1D NumPy array of RG‑flow values (coupling vs. scale).
    """
    return np.load(path)


def load_target_beta(path: str = None) -> np.ndarray:
    """
    Load a reference β‑function curve to compare against (e.g. CMB multipole data).
    If no path is provided, returns a placeholder target for sanity checks.

    Args:
      path: Optional path to a .npy file containing the target β‑function.

    Returns:
      A 1D NumPy array of target β values.
    """
    if path:
        return np.load(path)
    return np.linspace(1.0, 0.1, 100)


def compute_rmse(model: np.ndarray, target: np.ndarray) -> float:
    """
    Compute root‑mean‑square error between the model RG‑flow and a target curve.

    Args:
      model: 1D array of model β values.
      target: 1D array of target β values (must be same length as model).

    Returns:
      The RMSE as a float.
    """
    if model.shape != target.shape:
        raise ValueError(f"Shape mismatch: model {model.shape}, target {target.shape}")
    return float(np.sqrt(np.mean((model - target) ** 2)))


def plot_rg_flow(flow_path: str = "results/rg_flow.npy",
                 target_path: str = None,
                 output_path: str = "analysis/rg_flow.png") -> None:
    """
    Plot the RG‑flow beta function versus step and optionally compare to a target curve.
    Saves a PNG of the figure.

    Args:
      flow_path: Path to the model RG‑flow .npy file.
      target_path: Optional path to the reference β‑function .npy file.
      output_path: File path to save the plot image.
    """
    model = load_rg_flow(flow_path)
    steps = np.arange(model.size)

    plt.figure()
    plt.plot(steps, model, label="Model β-function")

    if target_path:
        target = load_target_beta(target_path)
        if target.shape != model.shape:
            x_old = np.linspace(0, model.size-1, target.size)
            target = np.interp(steps, x_old, target)
        plt.plot(steps, target, '--', label="Target β-function")

    plt.xlabel("RG Step")
    plt.ylabel("β-function value")
    plt.title("RG Flow vs. Target")
    plt.legend()
    plt.tight_layout()
    plt.savefig(output_path)
    plt.close()
    print(f"Saved RG flow plot to {output_path}")


def analyze_rg_flow(flow_path: str = "results/rg_flow.npy",
                    target_path: str = None) -> None:
    """
    Load and report on the RG‑flow results. If a target is provided, compute RMSE
    and generate a comparative plot.

    Args:
      flow_path: Path to the RG‑flow .npy file.
      target_path: Optional path to the reference β‑function .npy file.
    """
    model = load_rg_flow(flow_path)
    print(f"Loaded RG flow with {model.size} steps.")

    if target_path:
        target = load_target_beta(target_path)
        rmse = compute_rmse(model, target)
        print(f"RMSE vs. target: {rmse:.4f}")
    else:
        print("RG flow values:")
        print(model)

    # Always produce a plot for visualization
    plot_rg_flow(flow_path, target_path)


if __name__ == "__main__":
    analyze_rg_flow()
